(function () {
  'use strict';

  const buttons = document.querySelectorAll('[data-outside]');
  const ACTIVE_CLASS = 'is-active';
  function outsideClick(button) {
    if (!button) return;
    const target = document.getElementById(button.dataset.outside);
    if (!target) return;
    function toggleClasses() {
      button.classList.toggle(ACTIVE_CLASS);
      target.classList.toggle(ACTIVE_CLASS);
      if (button.classList.contains(ACTIVE_CLASS)) {
        document.addEventListener('click', clickOutside);
        return;
      }
      document.removeEventListener('click', clickOutside);
    }
    button.addEventListener('click', toggleClasses);
    function clickOutside(event) {
      if (!target.contains(event.target) && !button.contains(event.target)) {
        toggleClasses();
        document.removeEventListener('click', clickOutside);
      }
    }
    const closeButton = target.querySelector('[data-close]');
    if (closeButton) {
      closeButton.addEventListener('click', () => {
        button.classList.remove(ACTIVE_CLASS);
        target.classList.remove(ACTIVE_CLASS);
        document.removeEventListener('click', clickOutside);
      });
    }
  }
  function initOutsideClick() {
    buttons.forEach(button => {
      outsideClick(button);
    });
  }

  // Función para inicializar el lienzo (canvas)
  function initCanvas(container) {
    const canvas = document.createElement('canvas');
    canvas.setAttribute('id', 'visualizerCanvas');
    canvas.setAttribute('class', 'visualizer-item');
    container.appendChild(canvas);
    canvas.width = container.clientWidth;
    canvas.height = container.clientHeight;
    return canvas;
  }

  // Función para cambiar el lienzo según el tamaño del contenedor
  function resizeCanvas(canvas, container) {
    canvas.width = container.clientWidth;
    canvas.height = container.clientHeight;
  }

  // Visualizer
  const visualizer = (audio, container) => {
    if (!audio || !container) {
      return;
    }
    const options = {
      fftSize: container.dataset.fftSize || 2048,
      numBars: container.dataset.bars || 40,
      maxHeight: 80
    };
    const ctx = new AudioContext();
    const audioSource = ctx.createMediaElementSource(audio);
    const analyzer = ctx.createAnalyser();
    audioSource.connect(analyzer);
    audioSource.connect(ctx.destination);
    const frequencyData = new Uint8Array(analyzer.frequencyBinCount);
    const canvas = initCanvas(container);
    const canvasCtx = canvas.getContext('2d');

    // Crear barras
    const renderBars = () => {
      resizeCanvas(canvas, container);
      analyzer.getByteFrequencyData(frequencyData);
      if (options.fftSize) {
        analyzer.fftSize = options.fftSize;
      }
      canvasCtx.clearRect(0, 0, canvas.width, canvas.height);
      for (let i = 0; i < options.numBars; i++) {
        const index = Math.floor((i + 10) * (i < options.numBars / 2 ? 2 : 1));
        const fd = frequencyData[index];
        const barHeight = Math.max(4, fd / 255 * options.maxHeight);
        const barWidth = canvas.width / options.numBars;
        const x = i * barWidth;
        const y = canvas.height - barHeight;
        canvasCtx.fillStyle = 'white';
        canvasCtx.fillRect(x, y, barWidth - 2, barHeight);
      }
      requestAnimationFrame(renderBars);
    };
    renderBars();

    // Listener del cambio de espacio en la ventana
    window.addEventListener('resize', () => {
      resizeCanvas(canvas, container);
    });
  };

  const API_KEY_LYRICS = '1637b78dc3b129e6843ed674489a92d0';
  const cache = {};

  // Iconos de Meteor Icons: https://meteoricons.com/
  const icons = {
    play: '<svg width="24"  height="24" viewBox="0 0 24 24"  fill="currentColor"><path stroke="none" d="M0 0h24v24H0z" fill="none"/><path d="M6 4v16a1 1 0 0 0 1.524 .852l13 -8a1 1 0 0 0 0 -1.704l-13 -8a1 1 0 0 0 -1.524 .852z" /></svg>',
    pause: '<svg width="24"  height="24" viewBox="0 0 24 24"  fill="currentColor"><path stroke="none" d="M0 0h24v24H0z" fill="none"/><path d="M9 4h-2a2 2 0 0 0 -2 2v12a2 2 0 0 0 2 2h2a2 2 0 0 0 2 -2v-12a2 2 0 0 0 -2 -2z" /><path d="M17 4h-2a2 2 0 0 0 -2 2v12a2 2 0 0 0 2 2h2a2 2 0 0 0 2 -2v-12a2 2 0 0 0 -2 -2z" /></svg>',
    stop: '<svg width="24"  height="24" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M6 5H18C18.5523 5 19 5.44772 19 6V18C19 18.5523 18.5523 19 18 19H6C5.44772 19 5 18.5523 5 18V6C5 5.44772 5.44772 5 6 5Z"/></svg>'
  };
  const pixel = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAAAXNSR0IArs4c6QAAAA1JREFUGFdj+P//PxcACQYDCF0ysWYAAAAASUVORK5CYII=';

  // Convertir Unix timestamp a fecha
  function unixToDate(unix) {
    return new Date(unix * 1000);
  }

  // Convertir a tiempo relativo
  function timeAgo(date) {
    const getSecondsDiff = timestamp => (Date.now() - timestamp) / 1000;

    // Unidades de tiempo
    const DATE_UNITS = {
      day: 86400,
      hour: 3600,
      minute: 60,
      second: 1
    };

    // Obtener unidad y valor de la fecha
    const getUnitAndValueDate = secondsElapsed => {
      for (const [unit, secondsInUnit] of Object.entries(DATE_UNITS)) {
        if (secondsElapsed >= secondsInUnit || unit === 'second') {
          const value = Math.floor(secondsElapsed / secondsInUnit) * -1;
          return {
            value,
            unit
          };
        }
      }
    };

    // Obtener tiempo relativo
    const getTimeAgo = timestamp => {
      const rtf = new Intl.RelativeTimeFormat();
      const secondsElapsed = getSecondsDiff(timestamp);
      const {
        value,
        unit
      } = getUnitAndValueDate(secondsElapsed);
      return rtf.format(value, unit);
    };

    // Resultado
    const reference = new Date(date);
    return getTimeAgo(reference);
  }

  // Obtener porcentaje entre dos valores
  function getPercentage(value, total) {
    return value * 100 / total;
  }
  function convertSecondsToMinutes(seconds) {
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = seconds % 60;
    return `${minutes}:${remainingSeconds.toFixed(0) < 10 ? '0' : ''}${remainingSeconds.toFixed(0)}`;
  }

  // Obtener Datos desde Stream Africa v2
  // @param {string} query - Búsqueda
  // @param {string} service - Fuente de los datos
  const getDataRadioApi = async query => {
    const response = await fetch(`https://radioapi.lat/search?query=${query}`);
    const data = await response.json();

    // Si no responde
    if (data.results.length === 0) {
      const response = await fetch(`https://itunes.apple.com/search?term=${query}&entity=song&limit=1`);
      const data = await response.json();
      return data.results[0];
    }
    const results = data.results;
    return results;
  };

  // Obtener Datos de Canciones
  // @param {string} query - Búsqueda
  // @param {string} service - Fuente de los datos
  // @return {object} - Datos de la canción
  const getSongsData = async defaultData => {
    const query = `${defaultData.artist} ${defaultData.title}`;
    const key = `${query}`;

    // Si ya está en caché
    if (cache[key]) {
      return cache[key];
    }

    // Obtener datos si no está en caché
    const data = await getDataRadioApi(encodeURIComponent(query));
    const dataset = {
      ...defaultData,
      ...data
    };
    if (data.artworkUrl100) {
      const getArtwork = size => data.artworkUrl100.replace('100x100bb', `${size}x${size}bb`);
      dataset.artworkUrl100 = getArtwork('100');
      dataset.artworkUrl500 = getArtwork('500');
      dataset.artworkUrl1000 = getArtwork('1000');
      dataset.artworkUrl2000 = getArtwork('2000');
    } else {
      dataset.artworkUrl100 = data.artwork || pixel;
      dataset.artworkUrl500 = data.artwork || pixel;
      dataset.artworkUrl1000 = data.artwork || pixel;
      dataset.artworkUrl2000 = data.artwork || pixel;
    }

    // Guardar en caché
    cache[key] = dataset;
    return dataset;
  };

  // Establecer datos de la canción actual
  // @param {object} data - Datos de la canción
  // @param {object} hls - Instancia de HLS
  function getSongDefaultData(data, current) {
    return {
      artist: data.artist,
      title: data.title,
      playedAt: data.playedAt,
      duration: data.duration,
      elapsed: data.elapsed,
      remaining: data.remaining,
      artwork: current.picture || pixel,
      artworkUrl100: current.picture || pixel,
      artworkUrl500: current.picture || pixel,
      artworkUrl1000: current.picture || pixel,
      artworkUrl2000: current.picture || pixel
    };
  }

  // Obtener letras de canciones
  const getLyrics = async (artist, name) => {
    const response = await fetch(`https://api.vagalume.com.br/search.php?apikey=${API_KEY_LYRICS}&art=${encodeURIComponent(artist)}&mus=${encodeURIComponent(name)}`);
    const data = await response.json();
    if (data.type === 'exact' || data.type === 'aprox') {
      const lyrics = data.mus[0].text;
      return lyrics;
    }
    return 'Not found lyrics';
  };

  // Crear un elemento HTML a partir de una cadena de texto
  function createElementFromHTML(htmlString) {
    const div = document.createElement('div');
    div.innerHTML = htmlString.trim();
    return div.firstChild;
  }

  // Eliminar elementos innecesarios del texto
  function sanitizeText(text) {
    return text.replace(/^\d+\.\)\s/, '').replace(/<br>$/, '');
  }

  // Normalizar historial
  function normalizeHistory(api) {
    let artist;
    let title;
    let playedAt = false;
    let duration = false;
    let elapsed = false;
    let remaining = false;
    const history = api.song_history || api.history || api.songHistory || [];
    const historyNormalized = history.map(item => {
      if (api.song_history) {
        artist = item.song.artist;
        title = item.song.title;
        playedAt = item.played_at;
        duration = item.duration;
        elapsed = item.elapsed;
        remaining = item.remaining;
      } else if (api.history) {
        artist = sanitizeText(item.split(' - ')[0] || item);
        title = sanitizeText(item.split(' - ')[1] || item);
      } else if (api.songHistory) {
        artist = item.artist;
        title = item.title || item.song;
      }
      return {
        artist,
        title,
        playedAt,
        duration,
        elapsed,
        remaining
      };
    });
    return historyNormalized;
  }
  function createTempImage(src) {
    return new Promise((resolve, reject) => {
      const img = document.createElement('img');
      img.crossOrigin = 'Anonymous';
      img.src = `https://images.weserv.nl/?url=${src}`;
      img.onload = () => resolve(img);
      img.onerror = reject;
    });
  }
  function normalizeTitle(api) {
    let title;
    let artist;
    let playedAt = false;
    let duration = false;
    let elapsed = false;
    let remaining = false;
    if (api.songtitle) {
      title = api.songtitle.split(' - ')[1];
      artist = api.songtitle.split(' - ')[0];
    } else if (api.now_playing) {
      title = api.now_playing.song.title;
      artist = api.now_playing.song.artist;
      playedAt = api.now_playing.played_at;
      duration = api.now_playing.duration;
      elapsed = api.now_playing.elapsed;
      remaining = api.now_playing.remaining;
    } else if (api.artist && api.title) {
      title = api.title;
      artist = api.artist;
    } else if (api.currenttrack_title) {
      title = api.currenttrack_title;
      artist = api.currenttrack_artist;
    } else if (api.title && api.djusername) {
      title = api.title.split(' - ')[1];
      artist = api.title.split(' - ')[0];
    } else {
      title = api.currentSong;
      artist = api.currentArtist;
    }
    return {
      title,
      artist,
      playedAt,
      duration,
      elapsed,
      remaining
    };
  }

  // Devolver una promesa para saber si la imagen se ha cargado correctamente
  // @param {string} src - URL de la imagen
  // @returns {Promise} - Promesa que se resuelve si la imagen se carga correctamente
  function loadImage(src) {
    return new Promise((resolve, reject) => {
      const img = new Image();
      img.onload = () => resolve(img);
      img.onerror = reject;
      img.src = src;
    });
  }

  // Agrega una transición de deslizamiento a la imagen
  // @param {HTMLElement} container - Contenedor de la imagen
  // @param {string} src - URL de la imagen
  function slideUpImageTransition(container, src) {
    const img = document.createElement('img');
    const size = container.clientHeight;
    img.src = src;
    img.width = size;
    img.height = size;
    container.appendChild(img);
    const firstImg = container.querySelector('img:first-child');
    firstImg.style.marginLeft = `-${size}px`;
    firstImg.addEventListener('transitionend', () => {
      const allImgExcLast = container.querySelectorAll('img:not(:last-child)');
      allImgExcLast.forEach(img => img.remove());
    }, {
      once: true
    });
  }

  // Agrega una transición de desvanecimiento a la imagen
  // @param {HTMLElement} container - Contenedor de la imagen
  // @param {string} src - URL de la imagen
  function fadeImageTransition(container, src) {
    container.style.opacity = 0;
    container.addEventListener('transitionend', () => {
      container.src = src;
      container.style.opacity = 1;
      container.addEventListener('transitionend', () => {
        container.removeAttribute('style');
      }, {
        once: true
      });
    });
  }
  function setAccentColor(image, colorThief) {
    const dom = document.querySelector('html');
    const metaThemeColor = document.querySelector('meta[name=theme-color]');
    if (image.complete) {
      dom.setAttribute('style', `--accent: rgb(${colorThief.getColor(image)})`);
      metaThemeColor.setAttribute('content', `rgb(${colorThief.getColor(image)})`);
    } else {
      image.addEventListener('load', function () {
        dom.setAttribute('style', `--accent: rgb(${colorThief.getColor(image)})`);
        metaThemeColor.setAttribute('content', `rgb(${colorThief.getColor(image)})`);
      });
    }
  }

  // Establecer los elementos de la estación
  // @param {HTMLElement} container - Contenedor de la estación
  // @param {Object} data - Datos de la estación
  function setupStream(container, data) {
    if (!container || !data) return;
    const pictureEl = container.querySelector('.player-picture img:first-child');
    const backgroundEl = container.querySelector('.player-bg');
    const stationName = container.querySelector('.station-name');
    const streamName = container.querySelector('.stream-name');
    const streamDescription = container.querySelector('.stream-description');
    const stationTv = container.querySelector('.station-tv');
    const stationPicture = container.querySelector('.station-picture img');
    if (pictureEl) {
      // eslint-disable-next-line no-undef
      const colorThief = new ColorThief();
      createTempImage(data.picture).then(img => {
        setAccentColor(img, colorThief);
      });
      pictureEl.src = data.picture;
    }
    if (backgroundEl) {
      backgroundEl.src = data.background;
    }
    if (stationName) {
      stationName.textContent = data.name;
    }
    if (streamName) {
      streamName.textContent = data.name;
    }
    if (streamDescription) {
      streamDescription.textContent = data.description;
    }
    if (stationTv) {
      stationTv.classList.remove('none');
    }
    if (stationPicture) {
      stationPicture.src = data.picture;
    }
  }

  // Obtener el valor que corresponde al dia de la semana
  function getDayValue(item) {
    const day = new Date().getDay();
    switch (day) {
      case 1:
        return item.prog_horario_lu;
      case 2:
        return item.prog_horario_ma;
      case 3:
        return item.prog_horario_mi;
      case 4:
        return item.prog_horario_ju;
      case 5:
        return item.prog_horario_vi;
      case 6:
        return item.prog_horario_sa;
      case 0:
        return item.prog_horario_do;
      default:
        return '';
    }
  }

  // Función para obtener el día actual en formato de 2 letras
  function getCurrentDay() {
    const weekDays = ['do', 'lu', 'ma', 'mi', 'ju', 'vi', 'sa'];
    const day = new Date().getDay(); // 0 es domingo, 1 es lunes, etc.
    return weekDays[day];
  }
  function filterScheduleByDay(programs) {
    const currentDay = getCurrentDay();

    // Definir las claves de días a eliminar
    const clavesDias = ['prog_lu', 'prog_ma', 'prog_mi', 'prog_ju', 'prog_vi', 'prog_sa', 'prog_do'];
    const clavesHorario = ['prog_horario_lu', 'prog_horario_ma', 'prog_horario_mi', 'prog_horario_ju', 'prog_horario_vi', 'prog_horario_sa', 'prog_horario_do'];
    programs.filter(program => program['prog_' + currentDay] === '1');

    // Eliminar las claves de horario que no correspondan al día actual
    programs.forEach(program => {
      clavesHorario.forEach(clave => {
        if (clave !== 'prog_horario_' + currentDay) {
          delete program[clave];
        }
      });
    });

    // Eliminar los elementos del arreglo que no tengan horario
    programs = programs.filter(program => program['prog_horario_' + currentDay]);

    // Eliminar todas las claves de días
    programs.forEach(program => {
      clavesDias.forEach(clave => {
        delete program[clave];
      });
    });

    // Ordenar los programas por hora de menor a mayor
    programs.sort((a, b) => {
      const timeA = a['prog_horario_' + currentDay];
      const timeB = b['prog_horario_' + currentDay];
      return parseTime(timeA) - parseTime(timeB);
    });
    return programs;
  }

  // Función para convertir una hora "HH:mm" en minutos desde la medianoche
  function parseTime(time) {
    const [hours, minutes] = time.split(':').map(Number);
    return hours * 60 + minutes;
  }

  // Compara dos horas en formato HH:MM.
  // @param {string} hora1 - La primera hora.
  // @param {string} hora2 - La segunda hora.
  // @returns {string} - Indica si la primera hora es mayor, menor o igual a la segunda.
  function compareHours(hora, type, test) {
    // Obtener la hora actual, formateada en 24 horas, Ingles
    const currentTime = new Date().toLocaleTimeString('en-US', {
      hour: '2-digit',
      minute: '2-digit',
      hour12: false
    });

    // Fecha base arbitraria para crear objetos Date
    const base = '1970-01-01';
    const current = new Date(`${base}T${test || currentTime}:00`);
    const target = new Date(`${base}T${hora}:00`);
    if (type === 'greaterOrEqual') {
      return target >= current;
    }
    if (type === 'less') {
      return target < current;
    }
    if (type === 'equal') {
      return target === current;
    }
    return false;
  }
  function setupSchedule(container, data, baseUrl) {
    const programList = document.getElementById('program-list');
    if (!container || !data.length) return;
    const programs = filterScheduleByDay(data);
    programList.innerHTML = '';
    programs.forEach(item => {
      const li = document.createElement('li');
      li.classList.add('schedule-item', 'flex', 'g-1', 'items-center');
      li.innerHTML = `
      <div class="schedule-item-picture flex-none">
        <img src="${baseUrl}${item.prog_foto}" alt="${item.prog_titulo}">
      </div>
      <div class="schedule-item-content truncate">
        <h3 class="schedule-item-title truncate">${item.prog_titulo}</h3>
        <p class="schedule-item-description truncate">${getDayValue(item)} - ${item.prog_descripcion}</p>
      </div>
    `;
      programList.appendChild(li);
    });
    container.innerHTML = '';

    // Ordenar la programación por hora en formato 24 horas
    programs.sort((a, b) => {
      const aTime = getDayValue(a);
      const bTime = getDayValue(b);
      return aTime.localeCompare(bTime);
    });
    const filteredPrograms = [];
    for (let i = programs.length - 1; i >= 0; i--) {
      const programTime = getDayValue(programs[i]);

      // const test = '12:01'

      if (compareHours(programTime, 'greaterOrEqual')) {
        filteredPrograms.unshift(programs[i]);
      }
      if (compareHours(programTime, 'less')) {
        filteredPrograms.unshift(programs[i]);
        break;
      }
    }
    filteredPrograms.slice(0, 2).forEach((item, index) => {
      const li = document.createElement('li');
      li.classList.add('schedule-item', 'flex', 'g-1');
      li.innerHTML = `
      <div class="schedule-item-picture flex-none">
        <img src="${baseUrl}${item.prog_foto}" alt="${item.prog_titulo}">
      </div>
      <div class="schedule-item-content truncate">
        ${index === 0 ? '<span class="player-program-badge"><i></i> al aire</span>' : ''}
        ${index === 1 ? '<span class="player-program-text"><i></i> a continuacion</span>' : ''}
        <h3 class="schedule-item-title truncate">${item.prog_titulo}</h3>
        <p class="schedule-item-description truncate">${getDayValue(item)} - ${item.prog_descripcion}</p>
      </div>
    `;
      container.appendChild(li);
    });
  }

  // Iniciar la aplicación
  function initApp() {
    if (window.stream_active) return;
    window.stream_active = true;
    initOutsideClick();
    function play(audio, newSource = null) {
      if (newSource) {
        audio.src = newSource;
      }

      // Visualizer
      if (!hasVisualizer) {
        visualizer(audio, visualizerContainer);
        hasVisualizer = true;
      }
      audio.load();
      audio.play();
      playButton.innerHTML = icons.pause;
      playButton.classList.add('is-active');
    }
    function pause(audio) {
      audio.pause();
      playButton.innerHTML = icons.play;
      playButton.classList.remove('is-active');
    }

    // Botón play/pause, al pausar detener el stream, al reproducir iniciar el stream de nuevo
    // playButton, play, pause son funciones exportadas que se usaran en otros archivos
    const range = document.querySelector('.player-volume');
    const rangeFill = document.querySelector('.player-range-fill');
    const rangeWrapper = document.querySelector('.player-range-wrapper');
    const rangeThumb = document.querySelector('.player-range-thumb');
    let currentVolume;
    try {
      currentVolume = localStorage.getItem('volume') || 100;
    } catch (error) {
      currentVolume = 100;
      console.log('Error al obtener el volumen del localStorage:', error);
    }
    let currentStation;
    let currentActiveStation;
    let currentSongData = {};
    let customSongIsPlaying = false;
    let currentCustomEl;

    // Rango recorrido
    function setRangeWidth(percent) {
      {
        rangeFill.style.height = `${percent}%`;
      }
    }

    // Posición del thumb
    function setThumbPosition(percent) {
      const compensatedWidth = rangeWrapper.offsetHeight - rangeThumb.offsetHeight ;
      const thumbPosition = percent / 100 * compensatedWidth;
      {
        rangeThumb.style.bottom = `${thumbPosition}px`;
      }
    }

    // Actualiza el volumen al cambiar el rango
    function updateVolume(audio, value) {
      range.value = value;
      setRangeWidth(value);
      setThumbPosition(value);
      audio.volume = value / 100;
      if (localStorage === null || localStorage === undefined) return;
      localStorage.setItem('volume', value);
    }
    const playButton = document.querySelector('.player-button-play');
    const visualizerContainer = document.querySelector('.visualizer');
    let hasVisualizer = false;
    const audio = new Audio();
    audio.crossOrigin = 'anonymous';
    if (playButton !== null) {
      playButton.addEventListener('click', async () => {
        if (audio.paused) {
          play(audio);
        } else {
          pause(audio);
        }
      });
    }

    // Valor inicial
    if (range !== null) {
      updateVolume(audio, currentVolume);

      // Escucha el cambio del rango
      range.addEventListener('input', event => {
        updateVolume(audio, event.target.value);
      });

      // Escucha el movimiento del mouse
      rangeThumb.addEventListener('mousedown', () => {
        document.addEventListener('mousemove', handleThumbDrag);
      });
    }

    // Mueve el thumb y actualiza el volumen
    function handleThumbDrag(event) {
      const rangeRect = range.getBoundingClientRect();
      const click = event.clientY - rangeRect.top;
      let percent = click / range.offsetWidth * 100;
      percent = 100 - percent;
      percent = Math.max(0, Math.min(100, percent));
      const value = Math.round((range.max - range.min) * (percent / 100)) + parseInt(range.min);
      updateVolume(audio, value);
    }

    // Deja de escuchar el movimiento del mouse
    document.addEventListener('mouseup', () => {
      document.removeEventListener('mousemove', handleThumbDrag);
    });
    window.addEventListener('resize', () => {
      const currentPercent = range.value;
      setRangeWidth(currentPercent);
      setThumbPosition(currentPercent);
    });

    // ----------------------------------------------
    // Resto del código
    // ----------------------------------------------
    const appSuper = document.getElementById('app-super');
    const player = document.getElementById('super-player');
    const playerExpandButton = document.getElementById('player-expand');
    const hiddenPlayerButton = document.getElementById('player-down-up');
    const tvButton = document.querySelector('.player-button-tv');
    if (hiddenPlayerButton) {
      hiddenPlayerButton.addEventListener('click', () => {
        if (player.classList.contains('is-hidden')) {
          player.classList.remove('is-hidden');
        } else {
          player.classList.toggle('is-hidden');
        }
      });
    }
    if (playerExpandButton) {
      playerExpandButton.addEventListener('click', () => {
        if (appSuper.classList.contains('is-expanded')) {
          document.body.style.overflow = 'auto';
          playerExpandButton.classList.remove('is-open');
          appSuper.classList.remove('is-expanded');
        } else {
          document.body.style.overflow = 'hidden';
          playerExpandButton.classList.add('is-open');
          appSuper.classList.add('is-expanded');
        }
      });
    }
    const songNow = document.querySelector('.song-now');
    const lyricsContent = document.getElementById('lyrics');
    const history = document.getElementById('historyp');
    const historyTemplate = `<div class="historyp-item flex items-center g-0.75">
  <div class="historyp-image flex-none">
    <img src="{{art}}" width="80" height="80">
  </div>
  <div class="historyp-body truncate-line">
    <span class="historyp-name color-title fw-500">{{song}}</span>
    <span class="historyp-time color-text" data-time-ago="{{timeAgo}}">{{artist}}</span>
  </div>
</div>`;
    let currentVideo;
    let shareFullText = '';
    let observer;
    const TIME_TO_REFRESH = window?.streams?.timeRefresh || 10000;
    function songProgress(duration, elapsed) {
      const progress = getPercentage(elapsed, duration);

      // Agregar la variable css con el progreso de la canción
      appSuper.style.setProperty('--song-progress', `${2 + progress}%`);

      // Actualizar el progreso de la canción
      const songElapsed = document.querySelector('.song-elapsed');
      const songDuration = document.querySelector('.song-duration');
      if (songElapsed) {
        songElapsed.textContent = convertSecondsToMinutes(elapsed);
      }
      if (songDuration) {
        songDuration.textContent = convertSecondsToMinutes(duration);
      }
    }
    function setMediaSession(data) {
      const {
        title,
        artist,
        album,
        artworkUrl100,
        artworkUrl500
      } = data;
      if ('mediaSession' in navigator) {
        navigator.mediaSession.metadata = new MediaMetadata({
          title,
          artist,
          album,
          artwork: [{
            src: artworkUrl100,
            sizes: '128x128',
            type: 'image/png'
          }, {
            src: artworkUrl500,
            sizes: '256x256',
            type: 'image/png'
          }, {
            src: artworkUrl500,
            sizes: '512x512',
            type: 'image/png'
          }]
        });
        navigator.mediaSession.setActionHandler('play', () => {
          play(audio);
        });
        navigator.mediaSession.setActionHandler('pause', () => {
          pause(audio);
        });
      }
    }
    const videoContainer = document.getElementById('modal-body-video');
    function initVideoPlayer(station) {
      const $video = document.createElement('video');
      $video.id = 'player';
      $video.classList.add('video-js', 'video-js-stream');
      videoContainer.innerHTML = '';
      videoContainer.appendChild($video);

      // eslint-disable-next-line no-undef
      const video = videojs($video, {
        sources: [{
          src: station.tv_url,
          type: 'application/x-mpegURL'
        }],
        controls: true,
        autoplay: true
      });
      currentVideo = video;
    }
    function setTvButton(tvButton, station) {
      const tvModal = document.getElementById('modal-tv');
      const tvModalClose = tvModal.querySelector('[data-close]');
      if (tvButton) {
        tvButton.onclick = () => {
          if (tvButton.classList.contains('is-active')) {
            play(audio);
            tvModal.classList.remove('is-active');
            tvButton.classList.remove('is-active');
            if (currentVideo) {
              currentVideo.dispose();
            }
            return;
          }
          pause(audio);
          tvModal.classList.add('is-active');
          tvButton.classList.add('is-active');
          initVideoPlayer(station);
        };
      }
      if (tvModalClose) {
        tvModalClose.onclick = () => {
          if (currentVideo) {
            currentVideo.dispose();
            play(audio);
          }
          tvModal.classList.remove('is-active');
          tvButton.classList.remove('is-active');
        };
      }
    }
    function playerCustomPlay() {
      const playButton = document.querySelector('.player-custom-play');
      if (playButton) {
        playButton.addEventListener('click', event => {
          event.preventDefault();
          if (!audio.paused) {
            return;
          }
          play(audio);
        });
      }
    }
    function scrollText(container) {
      const item = container.querySelector('.song-title');
      if (item.scrollWidth > container.offsetWidth) {
        item.classList.add('is-scrolling');
        const scroll = item.scrollWidth - container.offsetWidth;
        const duration = 5000 + scroll * 100;
        item.setAttribute('style', `--text-scroll-duration: ${duration}ms; --text-scroll: -${scroll}px`);
        return;
      }
      item.classList.remove('is-scrolling');
      item.removeAttribute('style');
    }
    function normalizeCustomSong(data, baseUrl) {
      /* eslint-disable camelcase */
      const {
        pod_titulo,
        pod_desc,
        pod_url,
        pod_imagen
      } = data;
      const songData = {
        mp3: pod_url,
        title: pod_titulo,
        artist: pod_desc,
        artworkUrl100: `${baseUrl}${pod_imagen}`,
        artworkUrl500: `${baseUrl}${pod_imagen}`,
        artworkUrl1000: `${baseUrl}${pod_imagen}`
      };
      /* eslint-enable camelcase */

      return songData;
    }
    function updateSongProgress() {
      songProgress(audio.duration, audio.currentTime);
    }
    function playCustomSong(el, data) {
      // Volver a la estación original que estaba sonando
      if (customSongIsPlaying && currentCustomEl === el) {
        el.innerHTML = icons.play;
        audio.src = currentStation.stream_url;
        setCurrentSong(currentSongData);
        setMediaSession(currentSongData);
        play(audio);
        customSongIsPlaying = false;
        audio.removeEventListener('timeupdate', updateSongProgress);
        el.parentNode.classList.remove('is-playing');
        return;
      }
      const topsButtons = document.querySelectorAll('.nexo-tops-button');
      topsButtons.forEach(button => {
        button.innerHTML = icons.play;
        currentCustomEl?.parentNode.classList.remove('is-playing');
      });
      el.innerHTML = icons.stop;
      currentCustomEl = el;

      // agregar al padre del "el" la clase "is-playing"
      el.parentNode.classList.add('is-playing');
      setCurrentSong(data);
      setMediaSession(data);
      audio.src = data.mp3;
      audio.addEventListener('timeupdate', updateSongProgress);
      play(audio);
      customSongIsPlaying = true;
      /* eslint-enable camelcase */
    }
    function setCurrentSong(data) {
      const {
        title,
        artist,
        artworkUrl100,
        artworkUrl500,
        artworkUrl1000
      } = data;
      const content = songNow;
      const songName = document.querySelectorAll('.song-name');
      const songArtist = document.querySelectorAll('.song-artist');
      const socialFacebook = document.querySelector('.social-facebook');
      const socialTwitter = document.querySelector('.social-twitter');
      const socialWhatsapp = document.querySelector('.social-whatsapp');
      const website = window.location.origin;
      socialFacebook.href = `https://www.facebook.com/sharer/sharer.php?u=${website}`;
      socialTwitter.href = `https://twitter.com/intent/tweet?text=${shareFullText}`;
      socialWhatsapp.href = `https://api.whatsapp.com/send?text=${shareFullText}`;
      songName.forEach(item => {
        item.textContent = title;
      });
      songArtist.forEach(item => {
        item.textContent = artist;
      });
      const songTitle = content.querySelector('.player-title-fix');
      scrollText(songTitle);
      if (observer) {
        observer.disconnect();
      }
      observer = new ResizeObserver(() => {
        scrollText(songTitle);
      });
      observer.observe(songTitle);
      const pictureEl = content.querySelector('.player-artwork');
      const artworkUrl = artworkUrl1000 || artworkUrl500 || artworkUrl100;
      const pictureModal = document.querySelector('.player-modal-image');
      const playerLeft = document.querySelector('.player-left');
      playerLeft.style.removeProperty('--artwork');
      playerLeft.style.setProperty('--artwork', `url(${artworkUrl})`);
      if (pictureModal) {
        pictureModal.src = artworkUrl;
      }
      if (pictureEl) {
        loadImage(artworkUrl).then(() => slideUpImageTransition(pictureEl, artworkUrl)).catch(() => {
          console.error('Error al cargar arte de la canción');
        });
      }
      const coverEl = document.querySelector('.player-bg');
      if (coverEl) {
        const coverUrl = artworkUrl1000 || artworkUrl500 || artworkUrl100;
        const $img = document.createElement('img');
        $img.src = coverUrl;

        // eslint-disable-next-line no-undef
        const colorThief = new ColorThief();
        createTempImage($img.src).then(img => {
          setAccentColor(img, colorThief);
        });
        loadImage(coverUrl).then(() => fadeImageTransition(coverEl, coverUrl)).catch(() => {
          console.error('Error al cargar la portada de la canción');
        });
      }
    }
    function forcePlay(audio) {
      if (audio.paused) {
        play(audio);
      }
    }

    // Establecer las canciones que se han reproducido
    function setHistory(data, defaultData) {
      if (!history) return;
      history.innerHTML = historyTemplate.replace('{{art}}', pixel).replace('{{song}}', 'Cargando historial...').replace('{{artist}}', 'Artista').replace('{{timeAgo}}', 'time-none');
      if (!data) return;
      const promises = data.map(async item => {
        const {
          artist,
          title,
          playedAt,
          elapsed
        } = item;
        const dataFrom = await getSongsData({
          title,
          artist
        });
        const newHistoryTemplate = historyTemplate.replace('{{art}}', dataFrom.artworkUrl100 ?? (defaultData.artwork || pixel)).replace('{{song}}', dataFrom.title).replace('{{artist}}', dataFrom.artist).replace('{{timeAgo}}', 'time-none');
        if (elapsed) {
          return newHistoryTemplate.replace('time-none', 'Ahora');
        }
        if (playedAt) {
          return newHistoryTemplate.replace('time-none', timeAgo(unixToDate(playedAt)));
        }
        return newHistoryTemplate;
      });
      Promise.all(promises).then(itemsHTML => {
        const $fragment = document.createDocumentFragment();
        itemsHTML.forEach(itemHTML => {
          $fragment.appendChild(createElementFromHTML(itemHTML));
        });
        history.innerHTML = '';
        history.appendChild($fragment);
      }).catch(error => {
        console.error('Error:', error);
      });
    }
    function setLyrics(artist, title) {
      if (!lyricsContent) return;
      getLyrics(artist, title).then(lyrics => {
        const $p = document.createElement('p');
        $p.innerHTML = lyrics.replace(/\n/g, '<br>');
        lyricsContent.innerHTML = '';
        lyricsContent.appendChild($p);
      }).catch(error => {
        console.error('Error:', error);
      });
    }
    function loadTops(url) {
      window.mixxFunctions.dataTops = url;
      const dataTopsEl = document.getElementById('player-tops');
      if (!dataTopsEl) return;
      fetch(url).then(response => response.json()).then(data => {
        const template = `<div class="nexo-tops-item">
          <div class="nexo-tops-head">
            <div class="nexo-tops-image-container">
              <span>{{number}}</span>
              <img src="{{img}}" alt="imagen" class="nexo-tops-image" width="100">
            </div>
            <div class="nexo-tops-data">
              <div class="nexo-tops-title">{{title}}</div>
              <span class="nexo-tops-artist">{{artist}}</span>
            </div>
          </div>
          <div class="nexo-tops-footer">
            <div class="nexo-tops-footer-content">
              <button class="nexo-tops-button">${icons.play}</button>
              <div class="nexo-tops-meta">
                <div class="nexo-tops-meta-title">{{title}}</div>
                <div class="nexo-tops-meta-artist">{{artist}}</div>
              </div>
            </div>
          </div>
        </div>`;
        const $fragment = document.createDocumentFragment();
        data.forEach((item, index) => {
          const imgUrl = `${baseUrl}${item.pod_imagen}`;
          const newTemplate = template.replace('{{number}}', index + 1).replace('{{img}}', imgUrl).replace(/{{title}}/g, item.pod_titulo).replace(/{{artist}}/g, item.pod_desc);
          const $item = createElementFromHTML(newTemplate);
          const $button = $item.querySelector('.nexo-tops-button');
          const songData = normalizeCustomSong(item, baseUrl);
          $button.addEventListener('click', () => {
            playCustomSong($button, songData);
          });
          $fragment.appendChild($item);
        });
        dataTopsEl.innerHTML = '';
        dataTopsEl.appendChild($fragment);
        const $elements = document.querySelectorAll('.nexo-tops-head');
        let currentElement;
        $elements.forEach(element => {
          element.addEventListener('click', () => {
            if (currentElement === element) {
              element.classList.remove('is-active');
              currentElement = null;
              return;
            }
            if (currentElement) {
              currentElement.classList.remove('is-active');
            }
            element.classList.toggle('is-active');
            currentElement = element;
          });
        });
      }).catch(error => {
        console.error('Error:', error);
      });
    }
    function loadVideoTops(url) {
      window.mixxFunctions.dataTops = url;
      const dataTopsEl = document.getElementById('player-video-tops');
      if (!dataTopsEl) return;
      const listContainer = dataTopsEl.querySelector('.nexo-vid-tops-list');
      const videoContainer = dataTopsEl.querySelector('.nexo-vid-tops-video');
      if (!dataTopsEl) return;
      let currentElement;
      let currentVideo; // Declaramos aquí para que sea accesible en todo el contexto.

      function setVideo(url, thumbnail, play = false) {
        const $video = document.createElement('video');
        if (currentVideo && play) {
          currentVideo.pause();
          currentVideo.dispose(); // Eliminamos el video anterior
        }
        $video.id = 'player';
        $video.poster = thumbnail;
        $video.classList.add('video-js', 'video-js-stream');
        videoContainer.innerHTML = '';
        videoContainer.appendChild($video);

        // eslint-disable-next-line no-undef
        const video = videojs($video, {
          sources: [{
            src: url
          }],
          controls: true
        });
        currentVideo = video;
        video.ready(() => {
          if (play) {
            setTimeout(() => {
              video.play()?.then(() => {
                console.log('Reproduciendo video');
                if (appPlayer) pause(audio);
              }).catch(() => {
                console.error('Error al reproducir video');
              });
            }, 500);
          }
          video.on('play', () => {
            if (appPlayer) pause(audio);
          });
          video.on('pause', () => {
            if (appPlayer) forcePlay(audio);
          });
        });
      }
      fetch(url).then(response => response.json()).then(data => {
        const template = `<div class="nexo-vid-tops-item">
          <button class="nexo-vid-tops">
            <div class="nexo-vid-tops-image-container">
              <span>{{number}}</span>
              <img src="{{img}}" alt="imagen" class="nexo-vid-tops-image" width="140">
            </div>
            <div class="nexo-vid-tops-data">
              <div class="nexo-vid-tops-title">{{title}}</div>
              <span class="nexo-vid-tops-artist">{{description}}</span>
            </div>
          </button>
        </div>`;
        const $fragment = document.createDocumentFragment();
        data.forEach((item, index) => {
          const imgUrl = `${baseUrl}${item.photo}`;
          const vidUrl = `${baseUrl}${item.video}`;
          if (index === 0) {
            setVideo(vidUrl, imgUrl);
          }
          const newTemplate = template.replace('{{number}}', index + 1).replace('{{img}}', imgUrl).replace(/{{title}}/g, item.nombre).replace(/{{description}}/g, item.descripcion);
          const $item = createElementFromHTML(newTemplate);
          $fragment.appendChild($item);
        });
        listContainer.innerHTML = '';
        listContainer.appendChild($fragment);
        const $elements = document.querySelectorAll('.nexo-vid-tops');
        $elements.forEach((element, index) => {
          if (index === 0) {
            element.classList.add('is-active');
            currentElement = element;
          }
          element.addEventListener('click', () => {
            if (currentElement === element) {
              return;
            }
            if (currentElement) {
              currentElement.classList.remove('is-active');
            }
            element.classList.add('is-active');
            currentElement = element;
            const vidUrl = `${baseUrl}${data[index].video}`;
            const imgUrl = `${baseUrl}${data[index].photo}`;
            setVideo(vidUrl, imgUrl, true); // Llamamos a setVideo con "true" para que reproduzca automáticamente
          });
        });
      }).catch(error => {
        console.error('Error:', error);
      });
    }
    window.mixxFunctions = {
      loadTops,
      loadVideoTops
    };

    // --------------------------------------------------------------------------------------------//
    // INICIAR LA APLICACIÓN ----------------------------------------------------------------------//
    // --------------------------------------------------------------------------------------------//
    let currentSongPlaying;
    let timeoutId;
    let loadStations = [];
    const json = window.streams || {};
    const appPlayer = document.getElementById('app-player');
    const playerWrapper = document.querySelector('.player-wrapper');
    const hasLyrics = json.lyrics || false;
    const hasProgram = json.program || false;
    const hasExpand = json.expand || false;
    const hasShare = json.share_buttons || false;
    const baseUrl = json.base_url || '';
    const urlServer = json.url_server || 'api/';
    const idUser = json.id_user || '';
    const isMultiStream = json.multi_stream || false;
    const radioPath = isMultiStream ? 'multiradio-app/' : 'radio-app/';
    const dataUrl = `${baseUrl}${urlServer}${radioPath}${idUser}`;
    const dataProgramUrl = `${baseUrl}${urlServer}programa-app/${idUser}`;
    const dataTops = `${baseUrl}${urlServer}podcast-app/${idUser}`;
    const dataTopsVid = `${baseUrl}${urlServer}top-app/videos/${idUser}`;
    const buttonLyrics = document.querySelector('.player-button-lyrics');
    const buttonProgram = document.querySelector('.player-button-program');
    const buttonExpand = document.querySelector('.player-button-expand');
    const buttonPrev = document.querySelector('.player-button-prev');
    const buttonNext = document.querySelector('.player-button-next');
    const buttonShare = document.querySelector('.player-button-share');
    function enableButtons() {
      if (hasLyrics) {
        buttonLyrics.classList.add('is-loaded');
      }
      if (hasProgram) {
        buttonProgram.classList.add('is-loaded');
      }
      if (hasExpand) {
        buttonExpand.classList.add('is-loaded');
      }
      if (isMultiStream) {
        buttonPrev.classList.add('is-loaded');
        buttonNext.classList.add('is-loaded');
      }
      if (hasShare) {
        buttonShare.classList.add('is-loaded');
      }
      playerWrapper.addEventListener('touchmove', function (e) {
        e.preventDefault();
      }, {
        passive: false
      });
    }
    window.addEventListener('DOMContentLoaded', () => {
      loadVideoTops(dataTopsVid);
    });
    if (!appPlayer) return;
    enableButtons();
    fetch(dataUrl).then(response => response.json()).then(data => {
      if (isMultiStream) {
        const stations = data.map(item => {
          return {
            name: item.multiradio_nombre,
            description: item.multiradio_descripcion,
            stream_url: item.multiradio_url,
            api: item.multiradio_api,
            picture: `${baseUrl}${item.multiradio_imagen}`,
            background: pixel,
            tv_url: item.multiradio_tvstream,
            id: item.id
          };
        });
        loadStations = stations;
        setupStream(appPlayer, stations[0]);
        audio.src = stations[0].stream_url;
        if (stations[0].tv_url && stations[0].tv_url.includes('m3u8')) {
          tvButton.classList.add('is-loaded');
          setTvButton(tvButton, stations[0]);
        } else {
          tvButton.classList.remove('is-loaded');
        }
      } else {
        const station = {
          name: data.radio_nombre,
          stream_url: data.radio_url,
          tv_url: data.radio_video,
          picture: `${baseUrl}${data.radio_fondo}`,
          background: `${baseUrl}${data.radio_splash}`,
          api: data.radio_menu_nicio
        };
        currentStation = station;
        setupStream(appPlayer, station);
        audio.src = station.stream_url;
        if (station.tv_url && station.tv_url.includes('m3u8')) {
          tvButton.classList.add('is-loaded');
          setTvButton(tvButton, station);
        } else {
          tvButton.classList.remove('is-loaded');
        }
      }
      function init(current, change = false) {
        currentStation = current;
        if (timeoutId) clearTimeout(timeoutId);
        fetch(current.api).then(response => response.json()).then(async data => {
          if (data.now_playing && !customSongIsPlaying) {
            songProgress(data.now_playing.duration, data.now_playing.elapsed);
          }
          const song = normalizeTitle(data);
          if (currentSongPlaying !== song.title) {
            currentSongPlaying = song.title;
            const defaultData = getSongDefaultData(song, current);
            const historyData = normalizeHistory(data);
            const songData = await getSongsData(defaultData);
            currentSongData = songData;
            const newHistoryData = [{
              ...songData
            }, ...historyData];
            shareFullText = `Estoy escuchando ${songData.title} de ${songData.artist} en ${window.location.origin}`;
            setCurrentSong(songData);
            setMediaSession(songData);
            setHistory(data.now_playing ? newHistoryData : historyData, defaultData);
            if (hasLyrics) setLyrics(songData.artist, songData.title);
          }
        }).catch(error => console.log(error));
        if (hasProgram) {
          fetch(dataProgramUrl).then(response => response.json()).then(data => {
            const container = document.getElementById('player-programs');
            setupSchedule(container, data, baseUrl);
          }).catch(error => {
            console.error('Error:', error);
          });
        }
        timeoutId = setTimeout(() => {
          init(current);
        }, TIME_TO_REFRESH);
      }
      if (isMultiStream) {
        init(loadStations[0], true);
      } else {
        init(currentStation);
      }
      if (loadStations.length > 1) {
        const $stationsButton = document.querySelector('.player-stations');
        $stationsButton.classList.add('is-loaded');
        const $stations = document.getElementById('stations');
        if (!$stations) return;
        $stations.classList.add('is-visible');
        loadStations.forEach((station, index) => {
          const $button = document.createElement('button');
          $button.dataset.id = station.id;
          const $img = document.createElement('img');
          const $span = document.createElement('span');
          $img.src = station.picture;
          $img.alt = station.name;
          $img.classList.add('station-image', 'flex-none');
          $span.textContent = station.name;
          $button.classList.add('station-button', 'flex', 'items-center', 'g-0.5');
          $button.appendChild($img);
          $button.appendChild($span);
          if (index === 0) {
            $button.classList.add('is-current');
            currentActiveStation = $button;
          }
          $button.addEventListener('click', () => {
            if (currentStation === station) return;
            if (currentActiveStation) {
              currentActiveStation.classList.remove('is-current');
            }
            currentActiveStation = $button;
            currentActiveStation.classList.add('is-current');
            init(station, true);
            setupStream(appPlayer, station);
            currentStation = station;
            if (station.tv_url && station.tv_url.includes('m3u8')) {
              tvButton.classList.add('is-loaded');
              setTvButton(tvButton, station);
            } else {
              tvButton.classList.remove('is-loaded');
            }
            audio.src = station.stream_url;
            play(audio);
          });
          $stations.appendChild($button);
        });
        const $nextButton = buttonNext;
        const $prevButton = buttonPrev;
        $nextButton.addEventListener('click', () => {
          const $nextStation = currentActiveStation.nextElementSibling;
          if ($nextStation) {
            $nextStation.click();
          }
          if (!$nextStation && currentActiveStation) {
            const $firstStation = document.querySelector('.station-button');
            if ($firstStation) {
              $firstStation.click();
            }
          }
        });
        $prevButton.addEventListener('click', () => {
          const $prevStation = currentActiveStation.previousElementSibling;
          if ($prevStation) {
            $prevStation.click();
          }
          if (!$prevStation && currentActiveStation) {
            const $lastStation = document.querySelector('.station-button:last-child');
            if ($lastStation) {
              $lastStation.click();
            }
          }
        });
      }
    }).catch(error => {
      console.error('Error:', error);
    });
    window.addEventListener('DOMContentLoaded', () => {
      playerCustomPlay();
      loadTops(dataTops);
    });
  }
  initApp();
  if (typeof jQuery !== 'undefined') {
    // eslint-disable-next-line no-undef
    jQuery(document).on('ajaxComplete', function () {
      window.mixxFunctions.loadTops(window.mixxFunctions.dataTops);
      window.mixxFunctions.loadVideoTops(window.mixxFunctions.dataTopsVid);
    });
  }

})();
